' Tetris for CMM2
' Rev 1.0.0 William M Leue 7-Mar-2022
' Rev 4.0.0 5-Mar-2023
'  Added traditional 'Korobeiniki' background music
'  Fixed annoying delay after score increase.
option default integer
option base 1

const FPS          = 25
const WINDOWWIDTH  = mm.hres
const WINDOWHEIGHT = mm.vres
const BOXSIZE      = 20
const BOARDWIDTH   = 10
const BOARDHEIGHT  = 20
const BORDERWIDTH  = 7
const SCOREHEIGHT  = 60

const MOVESIDEWAYSFREQ! = 100.0
const MOVEDOWNFREQ! = 100.0
const INITIALFALLFREQ! = 750.0

const XMARGIN = (WINDOWWIDTH - BOARDWIDTH*BOXSIZE)\2
const TOPMARGIN = (WINDOWHEIGHT - BOARDHEIGHT*BOXSIZE)\2

const NEXT_X = 600
const NEXT_Y = 250
const NEXT_W = 7*BOXSIZE
const NEXT_H = 7*BOXSIZE

const NTEMPLATES     = 7
const MAXROTATIONS   = 4
const TEMPLATEWIDTH  = 5
const TEMPLATEHEIGHT = 5
const NCOLORS  = NTEMPLATES

const S_TEMPLATE     = 1
const Z_TEMPLATE     = 2
const I_TEMPLATE     = 3
const O_TEMPLATE     = 4
const J_TEMPLATE     = 5
const L_TEMPLATE     = 6
const T_TEMPLATE     = 7

const PTYPE  = 1
const PROT   = 2
const PCOL   = 3
const PROW   = 4
const NPVALS = 4

const EMPTY = 0
const NO_PIECE = 0

const MAXROWSPERLEVEL = 10

const MAXTOPSCORES = 10
const TS_W = 300
const TS_H = 300
const TS_X = mm.hres\2 - TS_W\2
const TS_Y = 200

const GO_W = 400
const GO_H = 200
const GO_X = mm.hres\2 - GO_W\2
const GO_Y = mm.vres\2 - GO_H\2

const UP = 128
const DOWN = 129
const LEFT = 130
const RIGHT = 131
const ESC = 27
const F1 = 145

const PTOG = asc("P")
const PTOG2 = asc("p")
const MUSIC$ = "Tetris.mp3"
const MUSIC_DURATION = 16500
const MUSIC_WAIT     = MUSIC_DURATION + 20000

' Globals
dim colors(NCOLORS)
dim NROTATIONS(NTEMPLATES) = (2, 2, 2, 1, 4, 4, 4)
dim templates(TEMPLATEWIDTH, TEMPLATEHEIGHT, MAXROTATIONS, NTEMPLATES)
dim scoreMultipliers(4) = (40, 100, 300, 1200)
dim fallingPiece(NPVALS)
dim prevFallingPiece(NPVALS)
dim nextPiece(NPVALS)

dim board(BOARDWIDTH, BOARDHEIGHT)
dim bdrowcount(BOARDHEIGHT)
dim lastMoveDownTime! = 0
dim lastMoveSidewaysTime! = 0
dim lastFallTime! = 0
dim movingDown = 0
dim movingLeft = 0
dim movingRight = 0
dim score = 0
dim level = 0
dim fallFreq! = 0.0
dim ticks = 0
dim rowsRemoved = 0

dim numTopScores = 0
dim TopScores(MAXTOPSCORES)
dim TopNames$(MAXTOPSCORES)

dim playing = 1

' Main program
open "debug.txt" for output as #1
ReadTemplates
ReadColors
ShowStartScreen
do
  PlayMusic
  settick MUSIC_WAIT, PlayMusic
  RunGame
  GameOver
  ShowStartScreen
loop
end

' Start the background music: the traditional 'Korobeiniki'
' folk song.
sub PlayMusic
  if playing then
    on error skip 1
    play stop
    play mp3 MUSIC$
  else
    play stop
  end if
end sub

' Read the 7 tetromino shape templates
sub ReadTemplates
  local type, rotation, row, col
  for type = 1 to NTEMPLATES
    for rotation = 1 to NROTATIONS(type)
      for row = 1 to TEMPLATEHEIGHT
        for col = 1 to TEMPLATEWIDTH
          read templates(col, row, rotation, type)
        next col
      next row
    next rotation
  next type
end sub

' Read the 7 piece colors
sub ReadColors
  local i, r, g, b
  for i = 1 to NCOLORS
    read colors(i)
  next i
end sub

' Show the Start Screen
sub ShowStartScreen
  local z$
  local cmd
  cls
  box 280, 2, 250, 30,, rgb(50, 50, 255), rgb(50, 50, 255)
  box 280, 32, 250, 30,, rgb(180, 180, 0), rgb(180, 180, 0)
  text mm.hres\2, 17, "Tetris", "CT", 5,, rgb(black), -1
  text mm.hres\2+2, 17, "Tetris", "CT", 5,, rgb(black), -1
  line 150, 70, 650, 70
  line 150, 100, 650, 100
  text mm.hres\2, 85, "Press F1 for Instructions, Escape to Quit, or any key to play", "CM"
  DrawTopScores
  z$ = INKEY$
  do : z$ = INKEY$ : loop until z$ <> ""
  cmd = asc(UCASE$(z$))
  if cmd = ESC then
    cls
    end
  end if
  if cmd = F1 then
    ShowHelpScreen
  end if
  cls
end sub

' Help for playing the game
sub ShowHelpScreen
  local z$
  cls
  text mm.hres\2, 20, "How to Play Tetris", "CT", 4,, rgb(green)
  text 0, 40, ""
  print "Tetris is a game in which you try to guide falling tetrominos so that they"
  print "completely fill rows of the board, Each time you manage to fill a row, it will be"
  print "erased from the board, and all the rows above it will move down. Simple, right?"
  print "Unfortunately for you, the falling random tetrominos will be coming faster and"
  print "faster. You can move the falling pieces left and right and rotate them  so that"
  print "they fit better, but as soon as your uncleared pieces reach the top of the board,"
  print "the game is over."
  print ""
  print "The scoring is as follows:"
  print "  Clearing a single row at a time:         40 * the current level+1
  print "  Clearing two rows at a time:            100 * the current level+1
  print "  Clearing three rows at a time:          300 * the current level+1
  print "  Clearing four rows at a time (TETRIS!) 1200 * the current level+1
  print ""
  print "Each time you have accumulated 10 cleared rows, the level increases by 1,"
  print "and the speed of the falling pieces increases."
  print ""
  print "The background music is the traditional Russian folksong 'Korobeiniki'. Press the
  print "P key to toggle the music off and on."
  print ""
  print "The controls are as follows:"
  print "  Move pieces LEFT:                      Left Arrow Key"
  print "  Move pieces RIGHT:                     Right Arrow Key"
  print "  Rotate pieces 90 degrees clockwise:    Up Arrow Key"
  print "  Drop a piece quickly:                  Down Arrow Key"
  print "  Toggle the music on or off:            P key
  print "  Quit the game:                         Escape Key"
  print ""
  print "There are 7 different tetromino pieces, which form the complete set of all possible"
  print "unique tetrominos."
  print ""
  print "(A note about the colors on the start page.)
  print "While I have no animus toward the Russian people, and in fact I admire Alexy Pajitnov,"
  print "the creator of Tetris, the current horrible war being waged against Ukraine is clearly"
  print "unjust, indefensible, and a threat to not only the people of Ukraine, but the entire"
  print "world. Therefore, the banner on the start page contains the colors of the Ukrainian"
  print "flag.   WML, March, 2022"
  text mm.hres\2, 500, "Press Any Key to Continue", "CT",,, rgb(green)
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' Top level for running the game
sub RunGame
  cls
  text mm.hres\2, 0, "Tetris!", "CT", 5,, rgb(red), -1
  InitBoard
  lastMoveDownTime! = 0
  lastMoveSidewaysTime! = 0
  lastFallTime! = 0
  Timer = 0
  movingDown = 0
  movingLeft = 0
  movingRight = 0
  score = 0
  level = 0
  rowsRemoved = 0
  CalculateLevelAndFallFreq()
  DrawStatus
  GetNewPiece fallingPiece()
  GetNewPiece nextPiece()
  DrawNextPiece nextPiece()
  GameLoop
end sub

' Game Over, Man!
sub GameOver
  local tsx(MAXTOPSCORES)
  local inits$
  box GO_X, GO_Y, GO_W, GO_H,, rgb(red), rgb(black)
  text GO_X+GO_W\2, GO_Y+50, "GAME OVER", "CT", 4,, rgb(red)
  numTopScores = ReadScores()
  if numTopScores > 0 then
    sort TopScores(), tsx(), 1, 1, numTopScores
  end if
  if numTopScores < MAXTOPSCORES or score > TopScores(1) then
    text GO_X+10, GO_Y+100, "You have a new top score!"
    text GO_X+10, GO_Y+130, "Enter your initials (up to 3 chars): "
    input "", inits$
    inc numTopScores
    topNames$(numTopScores) = UCASE$(LEFT$(inits$, 3))
    topScores(numTopScores) = score
    WriteScores numTopScores
  end if
  text GO_X+GO_W\2, GO_Y+170, "Press Any Key to Continue", "CT"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' Game loop: drop the piece, get user input and process
'  1. If the falling piece is invalid (NO_PIECE), make a new
'     falling piece from the 'next' piece and draw it.
'     If the new piece will not fit, exit the loop (Game Over).
'  2. Get the user's keyboard command, if any, and do the command:
'      LEFT - move falling piece left if possible.
'      RIGHT - move falling piece right if possible.
'      UP - rotate falling piece
'      DOWN - accellerate the falling piece speed.
'  3. Check the last time the falling piece fell against the timer,
'     and increment its position if enough time has elapsed.
'  4. If the falling piece has reached its resting place, erase it
'     and add its squares to the board. Mark the falling piece
'     invalid so a new one will be created (Step 1) on the next
'     loop pass.
'  5. If there are complete rows, erase them and update the score.
'    
sub GameLoop
  local cmd, type, rot, elapsed, nrl
  cmd = 0
  DrawBoard
  do
    if fallingPiece(PTYPE) = NO_PIECE then
      math add nextPiece(), 0, fallingPiece()
      GetNewPiece nextPiece()
      DrawNextPiece nextPiece()
      DrawPiece fallingPiece()
      math add fallingPiece(), 0, prevFallingPiece()
    end if
    type = fallingPiece(PTYPE)
    rot = fallingPiece(PROT)
    if not IsValidPosition(fallingPiece(), rot, 0, 0) then exit do
    cmd = GetUserInput()
    select case cmd
      case LEFT
        if IsValidPosition(fallingPiece(), rot, -1, 0) then
          elapsed = timer - lastMoveSidewaysTime!
          if elapsed >= MOVESIDEWAYSFREQ then
            inc fallingPiece(PCOL), -1
            AnimateFallingPiece
            movingLeft = 1
            movingRight = 0
            lastMoveSidewaysTime! = timer
          end if
        end if
      case RIGHT
        if IsValidPosition(fallingPiece(), rot, 1, 0) then
          elapsed = timer - lastMoveSidewaysTime!
          if elapsed >= MOVESIDEWAYSFREQ then
            inc fallingPiece(PCOL), 1
            AnimateFallingPiece
            movingLeft = 0
            movingRight = 1        
            lastMoveSidewaysTime! = timer
          end if
        end if
      case UP
        rot = fallingPiece(PROT)
        rot = (rot+1) 
        if rot > NROTATIONS(type) then rot = 1
        if IsValidPosition(fallingPiece(), rot, 0, 0) then
          fallingPiece(PROT) = rot
          AnimateFallingPiece
        end if
      case DOWN
        if IsValidPosition(fallingPiece(), rot, 0, 1) then
          elapsed = timer - lastMoveDownTime!
          if elapsed >= MOVEDOWNFREQ then
            inc fallingPiece(PROW), 1
            AnimateFallingPiece
            movingDown = 1
            lastMoveDownTime! = timer
          end if
        end if
      case PTOG, PTOG2
        playing = 1 - playing
        if playing then
          PlayMusic
        else
          play stop
        end if
      case ESC
        Quit
      case else
        ' pass
    end select
    elapsed = timer - lastFallTime!
    if elapsed >= fallFreq! then
      inc ticks
      if IsValidPosition(fallingPiece(), rot, 0, 1) then
        inc fallingPiece(PROW)
        lastFallTime! = timer
        AnimateFallingPiece
      else
        AddToBoard(fallingPiece())
        ErasePiece fallingPiece()
        DrawBoard
        fallingPiece(PTYPE) = NO_PIECE
        nrl = RemoveCompleteLines()
        inc rowsRemoved, nrl
        if nrl > 0 then
          inc score, (scoreMultipliers(nrl)*(level+1))
          CalculateLevelAndFallFreq()
          DrawStatus
        end if
      end if
    end if
    end if
  loop
end sub 

' Initialize the board for a new game
sub InitBoard
  local row, col
  for row = 1 to BOARDHEIGHT
    bdrowcount(row) = 0
    for col = 1 to BOARDWIDTH
      board(col, row) = 0
    next col
  next row
end sub

' Calculate level and falling speed
sub CalculateLevelAndFallFreq()
  if rowsRemoved >= MAXROWSPERLEVEL then
    inc level
    rowsRemoved = 0
  end if
  fallFreq! = INITIALFALLFREQ! - (level*100)
end sub

' Create a new random piece
sub GetNewPiece piece()
  local type
  type = RandInt(1, NTEMPLATES)
  if type < 1 or type > NTEMPLATES then
    print #1, "Piece type out of range: ";piece(PTYPE);" aborting"
    end
  end if
  piece(PTYPE) = type
  piece(PROT) = 1
  piece(PCOL) = BOARDWIDTH\2 - TEMPLATEWIDTH\2 - 1
  if type = I_TEMPLATE then
    piece(PROW) = 0
  else
    piece(PROW) = -1
  end if
end sub

' return 1 if a proposed piece placement on the board is legal,  else 0
' This function is used in two places:
'  At the top of the screen to detect game over.
'  At the bottom of the screen to detect then the falling piece has no
'  place to fall further.
function IsValidPosition(piece(), adjRot, adjCOL, adjROW)
  local type, rot, col, row, bcol, brow
  type = piece(PTYPE)
  for col = 1 to TEMPLATEWIDTH
    for row = 1 to TEMPLATEHEIGHT
      bcol = col + piece(PCOL) + adjCOL
      brow = row + piece(PROW) + adjROW
      if brow < 1 then continue for
      if templates(col, row, adjRot, type) = 1 then
        if not IsOnBoard(bcol, brow) then
          IsValidPosition = 0
          exit function
        else
          if (templates(col, row, adjRot, type)) > 0 then
            if board(bcol, brow) <> EMPTY then
              IsValidPosition = 0
              exit function
            end if
          end if
        end if
      end if
    next row
  next col
  IsValidPosition = 1
end function

' return 1 if a (col,row) location is on the board, 0 otherwise
function IsOnBoard(bcol, brow)
  local ok = 1
  if bcol < 1 or bcol > BOARDWIDTH then ok = 0
  if brow < 1 or brow > BOARDHEIGHT then ok = 0
  IsOnBoard = ok
end function

' Add a piece whose fall is over to the board.
' This no-longer-falling piece now gets drawn in its final position
' every time the board gets drawn.
sub AddToBoard(piece())
  local type, rot, col, row, bcol, brow
  type = piece(PTYPE)
  rot = piece(PROT)
  for col = 1 to TEMPLATEWIDTH
    for row = 1 to TEMPLATEHEIGHT
      if templates(col, row, rot, type) <> 0 then
        bcol = col + piece(PCOL)
        brow = row + piece(PROW)
        board(bcol, brow) = piece(PTYPE)
        inc bdrowcount(brow)
      end if
    next row
  next col
end sub

' Remove completed lines at the bottom of the board
' Returns the number of lines removed.
function RemoveCompleteLines()
  local y, num, col, row, pdrow
  num = 0
  y = BOARDHEIGHT
  do while y >= 1
    if IsCompleteLine(y) then
      inc num
      for pdrow = y to 2 step -1
        if bdrowcount(pdrow) > 0 then
          for col = 1 to BOARDWIDTH
            board(col, pdrow) = board(col, pdrow-1)
          next col
        end if
      next pdrow
      for col = 1 to BOARDWIDTH
        board(col, 1) = 0
      next col
      bdrowcount(1) = 0
    else
      inc y, -1
    end if
  loop
  if num > 0 then DrawBoard
  RemoveCompleteLines = num
end function

' Return 1 if the specified board row is completely filled in
function IsCompleteLine(row)
  local col
  IsCompletedLine = 0
  for col = 1 to BOARDWIDTH
    if board(col, row) = 0 then exit function
  next col
  IsCompleteLine = 1
end function

' Get user key input (CBreak mode)
function GetUserInput()
  local cmd = 0
  local z$ = ""
  z$ = INKEY$
  if z$ <> "" then
    cmd = asc(UCASE$(z$))
    GetUserInput = cmd
    z$ = ""
  end if
end function

' Pause the Game
sub PauseGame
end sub

' Quit the Game
sub Quit
  cls
  end
end sub

' Draw the board. This only draws the border and the
' fixed pieces.
sub DrawBoard
  local row, col
  local x, y, w, h
  local bcolor = rgb(60, 10, 10)
  x = XMARGIN-BORDERWIDTH
  y = TOPMARGIN-BORDERWIDTH
  w = BOARDWIDTH*BOXSIZE + 2*BORDERWIDTH
  h = BOARDHEIGHT*BOXSIZE + 2*BORDERWIDTH
  box x, y, w, h, BORDERWIDTH, bcolor
  box x, y-SCOREHEIGHT, w, SCOREHEIGHT, BORDERWIDTH, bcolor
  for col = 1 to BOARDWIDTH
    for row = 1 to BOARDHEIGHT
      DrawBox col, row, board(col, row))
    next row
  next col
end sub

' Draw a board cell. Unoccupied cells are black
' This only shows pieces that have fallen to the bottom
' and are frozen. Active falling pieces are drawn by
' the DrawPiece method.
sub DrawBox bcol, brow, type
  local x, y, c
  if type = NO_PIECE then
    c = rgb(black)
  else
    c = colors(type)
  end if
  x = XMARGIN + (bcol-1)*BOXSIZE
  y = TOPMARGIN + (brow-1)*BOXSIZE
  box x, y, BOXSIZE, BOXSIZE,, rgb(black), c
end sub

' Show the current level and score
sub DrawStatus()
  local x, y, bw
  v$ = "Level " + str$(level)
  s$ = "Score " + str$(score)
  bw = BOARDWIDTH*BOXSIZE
  x = XMARGIN
  y = TOPMARGIN-SCOREHEIGHT
  text x+bw\2, y+4, v$, "CT", 4,, rgb(yellow)
  text x+bw\2, y+25, s$, "CT", 4,, rgb(green)
end sub

' Erase the falling piece at its previous position
sub ErasePiece piece()
  local type, rot, col, row, x, y, bcol, brow, tcol, trow
  type = piece(PTYPE)
  rot = piece(PROT)
  tcol = piece(PCOL)
  trow = piece(PROW)
  if type = 0 or rot = 0 then exit sub
  for col = 1 to TEMPLATEWIDTH
    for row = 1 to TEMPLATEHEIGHT
      if templates(col, row, rot, type) <> 0 then
        bcol = col + tcol
        brow = row + trow
        DrawBox bcol, brow, rgb(black)
      end if
    next row
  next col
end sub

' Animate a falling piece by erasing the old position and
' redrawing it in the new position.
sub AnimateFallingPiece
  ErasePiece prevFallingPiece()
  DrawPiece fallingPiece()
  math add fallingPiece(), 0, prevFallingPiece()
end sub

' Draw the falling piece
' Only the falling piece is drawn: the ones that have already
' fallen into their final positions are drawn when the board
' is drawn.
sub DrawPiece piece()
  local type, rot, col, row, x, y, bcol, brow, tcol, trow
  type = piece(PTYPE)
  rot = piece(PROT)
  tcol = piece(PCOL)
  trow = piece(PROW)
  for col = 1 to TEMPLATEWIDTH
    for row = 1 to TEMPLATEHEIGHT
      if templates(col, row, rot, type) <> 0 then
        bcol = col + tcol
        brow = row + trow
        DrawBox bcol, brow, piece(PTYPE)
      end if
    next row
  next col
end sub

' Draw the Next Piece in the Box
sub DrawNextPiece piece()
  local row, col, x, y, type, rot
  type = piece(PTYPE)
  rot = piece(PROT)
  box NEXT_X, NEXT_Y, NEXT_W, NEXT_H,, rgb(white), rgb(black)
  text NEXT_X+NEXT_W\2, NEXT_Y-2, "Next Piece", "CB"
  for col = 1 to TEMPLATEWIDTH
    for row = 1 to TEMPLATEHEIGHT
      if templates(col, row, rot, type) <> 0 then
        x = NEXT_X + BOXSIZE + (col-1)*BOXSIZE
        y = NEXT_Y + BOXSIZE + (row-1)*BOXSIZE
        box x, y, BOXSIZE, BOXSIZE,, rgb(black), colors(type)
      end if
    next row
  next col
end sub

' Draw the top scores on the Start Screen
sub DrawTopScores
  local tsx(MAXTOPSCORES), i, x, y
  local s$
  text TS_X+TS_W\2, TS_Y-30, "TOP 10 SCORES", "CT", 4,, rgb(blue)
  text TS_X+TS_W\4, TS_Y-2, "Initials", "CB"
  text TS_X+150+TS_W\4, TS_Y-2, "Scores", "CB"
  box TS_X, TS_Y, TS_W, TS_H
  line TS_X+TS_W\2, TS_Y, TS_X+TS_W\2, TS_Y+TS_H-1
  numTopScores = ReadScores()
  if numTopScores  > 1 then
    sort TopScores(), tsx(), 1, 1, numTopScores
  else
    tsx(1) = 1
  end if
  if numTopScores > 0 then
    x = TS_X+50
    y = TS_Y+10
    for i = 1 to numTopScores
      s$ = str$(i) + ": " + TopNames$(tsx(i))
      text x, y, s$
      text x+150, y, str$(TopScores(i))
      inc y, 20
    next i
  else
    text TS_X+TS_W\2, TS_Y+TS_H\2, "No Top Scores yet", "CM"
  end if
end sub

' Read the top scores file (honor system :))
function ReadScores()
  local i, n, txs(MAXTOPSCORES)
  local buf$
  on error skip 1
  open "topscores.txt" for input as #2
  if mm.errno <> 0 then
    WriteScores 0
    ReadScores = 0
    exit function
  end if
  line input #2, buf$
  n = val(buf$)
  for i = 1 to n
    line input #2, buf$
    cat buf$, ","
    TopNames$(i) = field$(buf$, 1, ",")
    TopScores(i) = val(field$(buf$, 2, ","))
  next i
  close #2
  ReadScores = n
end function

' Write the top scores file
sub WriteScores n
  local i
  local buf$
  open "topscores.txt" for output as #2
  print #2, str$(n)
  for i = 1 to n
    buf$ = TopNames$(i) + "," + str$(TopScores(i))
    print #2, buf$
  next i
  close #2 
end sub

' return a uniformly distributed random integer in the specified closed range
function RandInt(a as integer, b as integer)
  local integer v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandInt = v
end function

' (debug)
sub PrintPiece piece()
  print #1, "Piece: "
  print #1, "  PTYPE:  ";piece(PTYPE)
  print #1, "  PROT:   ";piece(PROT)
  print #1, "  PCOL:   ";piece(PCOL)
  print #1, "  PROW:   ";piece(PROW)
end sub

' (debug)
sub PrintBoard
  local col, row
  for row = 1 to BOARDHEIGHT
    for col = 1 to BOARDWIDTH
      print #1, board(col, row);" ";
    next col
    print #1, ""
  next row
end sub    

' tetromino templates

' S tetromino (2 rotations)
data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0
data 0, 0, 1, 1, 0
data 0, 1, 1, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 1, 0
data 0, 0, 0, 1, 0
data 0, 0, 0, 0, 0

' Z tetromino (2 rotations)
data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0
data 0, 1, 1, 0, 0
data 0, 0, 1, 1, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 1, 0, 0
data 0, 1, 1, 0, 0
data 0, 1, 0, 0, 0
data 0, 0, 0, 0, 0

' I tetromino (2 rotations)
data 0, 0, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0
data 1, 1, 1, 1, 0
data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0

' O tetromino (1 rotation)
data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0
data 0, 1, 1, 0, 0
data 0, 1, 1, 0, 0
data 0, 0, 0, 0, 0

' J tetromino (4 rotations)
data 0, 0, 0, 0, 0
data 0, 1, 0, 0, 0
data 0, 1, 1, 1, 0
data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 1, 1, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0
data 0, 1, 1, 1, 0
data 0, 0, 0, 1, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 1, 1, 0, 0
data 0, 0, 0, 0, 0

' L tetromino (4 rotations)
data 0, 0, 0, 0, 0
data 0, 0, 0, 1, 0
data 0, 1, 1, 1, 0
data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 1, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0
data 0, 1, 1, 1, 0
data 0, 1, 0, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 1, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 0, 0, 0

' T tetromino (4 rotations)
data 0, 0, 0, 0, 0
data 0, 0, 1, 0, 0
data 0, 1, 1, 1, 0
data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 1, 1, 0
data 0, 0, 1, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 0, 0, 0
data 0, 1, 1, 1, 0
data 0, 0, 1, 0, 0
data 0, 0, 0, 0, 0

data 0, 0, 0, 0, 0
data 0, 0, 1, 0, 0
data 0, 1, 1, 0, 0
data 0, 0, 1, 0, 0
data 0, 0, 0, 0, 0


' Color RGB values
data rgb(70, 0, 255)
data rgb(red)
data rgb(green)
data rgb(255, 100, 0)
data rgb(blue)
data rgb(yellow)
data rgb(cyan)

